<?php

namespace App\Http\Controllers\API\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Auth;
use Socialite;
use Validator;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\Cache;
use Symfony\Component\HttpFoundation\Response;

class UserAPIController extends Controller
{
    private $language;
    public function __construct()
    {
        // No need to call parent::__construct() here
        $this->language = request()->header('language-code') && request()->header('language-code') != '' ? request()->header('language-code') : 'en';
    }

    /**
     *  Do Signup
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    function doSignup(Request $request)
    {
        $validator = Validator::make($request->all(), [
            // 'name' => 'required',
            // 'email' => 'required|email|unique:users',
            'phone' => ['required', 'unique:users', 'regex:/^\+\d{1,3}\d{6,14}$/'],
            'password' => 'required|min:8',
            'confirm_password' => 'required|same:password',
            'profile_pic' => 'image|max:2048',
            'device_type' => 'required|in:web,app',
        ]);

        if ($validator->fails()) {
            return $this->sendError(__('lang.message_required_message'),$validator->errors());
        }

        $userData = [
            'type' => 'user', // Added equals sign
            'device_type' => $request->device_type, // Added equals sign
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone, 
            'role_id' => '2',
            'token' => \Helpers::generateToken(),
            'status' => '1', // Corrected the variable to string
            'password' => Hash::make($request->confirm_password),
            'created_at' => date("Y-m-d H:i:s")
        ];

        if (isset($request['profile_pic']) && $request['profile_pic'] != '') {
            $uploadImage = \Helpers::uploadFiles($request['profile_pic'], 'user/profile_pic');
            if ($uploadImage['status'] == true) {
                $userData['image'] = $uploadImage['file_name']; // Changed 'image' to 'profile_pic'
            }
        }

        $user_id = User::insertGetId($userData); // Changed $user to $user_id

        if ($user_id) {
            $userData = User::where('id', $user_id)->first(); // Changed $user to $user_id
            if ($userData) {
                if ($userData->profile_pic != '') { // Changed 'image' to 'profile_pic'
                    $userData->profile_pic = url('uploads/user/profile_pic/' . $userData->profile_pic); // Changed 'image' to 'profile_pic'
                }
            }
            return $this->sendResponse($userData, __('lang.message_signup_success'));
        }
        return $this->sendError(__('lang.message_user_registered'));
    }

    /**
     *  Do Sign In
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    function doLogin(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email_phone' => 'required', // email or phone
            'password' => 'required',
        ]);

        if ($validator->fails()) {
            return $this->sendError(__('lang.message_required_message'), $validator->errors());
        }
        // Find user by email or phone
        $user = User::where(function($query) use ($request) {
                $query->where('email', $request->email_phone)
                      ->orWhere('phone', $request->email_phone);
            })
            ->where('type', 'user')
            ->where('id', '!=', 1)
            ->first();

        if (!$user) {
            return $this->sendError(__('lang.message_wrong_details'));
        }

        // Check if the password is correct
        if (!Hash::check($request->password, $user->password)) {
            return $this->sendError(__('lang.message_wrong_details'));
        }
        // Update the token
        $user->token = \Helpers::generateToken();
        $user->save();

        // Return user data with updated token
        $user->profile_pic = $user->profile_pic ? url('uploads/user/profile_pic/' . $user->profile_pic) : null;

        return $this->sendResponse($user, __('lang.message_login_success'));
    }

    /**
     *  Do Rediraction With Google
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function doRedirectToGoogle()
    {
        return Socialite::driver('google')->redirect();
    }

    /**
     *  Do Sign With Google
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function handleGoogleCallback()
    {
        try {
            $googleUser = Socialite::driver('google')->user();
            $user = User::where('email', $googleUser->email)->first();

            if ($user) {
                // User exists, update their token and other information
                $user->token = \Helpers::generateToken();
                $user->name = $googleUser->name;
                $user->profile_pic = $googleUser->avatar;
                $user->save();
            } else {
                // User does not exist, create a new user
                $user = User::create([
                    'type' => 'user',
                    'name' => $googleUser->name,
                    'email' => $googleUser->email,
                    'google_id' => $googleUser->id,
                    'profile_pic' => $googleUser->avatar,
                    'role_id' => '2',
                    'token' => \Helpers::generateToken(),
                    'status' => 'active',
                    'password' => Hash::make(uniqid()), // Generate a random password
                    'created_at' => now(),
                ]);
            }

            // Return user data with updated token
            $user->profile_pic = $user->profile_pic ? url('uploads/user/profile_pic/' . $user->profile_pic) : null;
            return $this->sendResponse($user, __('lang.message_signup_success'));

        } catch (\Exception $e) {
            // Log the error for debugging purposes
            \Log::error('Error during Google signup: ' . $e->getMessage());

            // Return an error response
            return response()->json(['error' => __('lang.message_google_signup_failed')], 500);
        }
    }



}
