<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Category;
use Session;
use App\Exports\CategoryExport;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Facades\Validator;
use App\Imports\CategoryImport;

class CategoryBulkController extends Controller
{

    // Display a listing of the categories.
    public function categoryBulk(Request $request)
    {
        try{
            return view('admin\bulk\category_export');
        }
        catch(\Exception $ex){
            return redirect()->back()->with('error', $ex->getMessage() . ' '. $ex->getLine() . ' '. $ex->getFile()); 
        }
    }

    // this is for export data
    public function export(Request $request)
    {
        $type = $request->type;

        if ($type == 'date_wise') {
            $fromDate = $request->input('from_date');
            $toDate = $request->input('to_date');

            $validator = Validator::make($request->all(), [
                'from_date' => 'required|date',
                'to_date' => 'required|date|after_or_equal:from_date',
            ]);

            if ($validator->fails()) {
                return redirect()->back()->with('error', 'Error: ' . implode(', ', $validator->errors()->all()));
            }


            $data = $this->getDataDateWise($fromDate, $toDate);
        } elseif ($type == 'id_wise') {
            $fromId = $request->input('from_id');
            $toId = $request->input('to_id');

            $validator = Validator::make($request->all(), [
                'from_id' => 'required|numeric',
                'to_id' => 'required|numeric|gte:from_id',
            ]);

            if ($validator->fails()) {
                return redirect()->back()->with('error', 'Error: ' . implode(', ', $validator->errors()->all()));
            }

            $data = $this->getDataIdWise($fromId, $toId);
        } else if($type == 'all_data'){
            $data = $this->getAllData();
        }

        $dateTime = now()->format('Y-m-d_H-i-s');
        $fileType = ucfirst($type);
        $customFileName = "category_export_{$dateTime}_{$fileType}.xlsx";

        // session()->flash('success', 'Export successful!');

        return Excel::download(new CategoryExport($data), $customFileName);
    }

   // this is for get all data
    private function getAllData()
    {
        return Category::all();
    }
    
    // this is for date wise
    private function getDataDateWise($fromDate, $toDate)
    {
        return Category::whereBetween('created_at', [$fromDate, $toDate])->get();
    }

    // this is for id wise
    private function getDataIdWise($fromId, $toId)
    {
        return Category::whereBetween('id', [$fromId, $toId])->get();
    }

    // this id for download sample file without data
    public function downloadSampleWithoutData()
    {
        $sampleFilePath = public_path('excel/category_sample_without_data.xlsx');
        $headers = [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ];

        return Response::download($sampleFilePath, 'category_sample_without_data.xlsx', $headers);
    }

    // this is for doenload sample file with data
    public function downloadSampleWithData()
    {
        $sampleFilePath = public_path('excel/category_sample_with_data.xlsx');
        $headers = [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ];

        return Response::download($sampleFilePath, 'category_sample_with_data.xlsx', $headers);
    }

    // this is for import file
    public function import(Request $request)
    {

        $validator = Validator::make($request->all(), [
             'import_file' => 'required|file|mimes:xlsx,xls',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->with('error', 'Error: ' . implode(', ', $validator->errors()->all()));
        }

        try {
            $import = new CategoryImport();
            $file = $request->file('import_file');

            // Import the data
            Excel::import($import, $file);

            return redirect()->back()->with('success', __('lang.admin_data_import_msg'));
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Error importing data. ' . $e->getMessage());
        }
    }


}
